function commonSubmitArgs = getCommonSubmitArgs(cluster, numWorkers)
% Get any additional submit arguments for the Slurm sbatch command
% that are common to both independent and communicating jobs.

% Copyright 2016-2022 The MathWorks, Inc.

% https://confluence.desy.de/display/MXW/Running+batch+jobs

commonSubmitArgs = '';
ap = cluster.AdditionalProperties;

% Number of cores/node
ppn = validatedPropValue(ap, 'ProcsPerNode', 'double', 0);
if ppn>0
    % Don't request more cores/node than workers
    ppn = min(numWorkers*cluster.NumThreads,ppn);
    assert(rem(ppn,cluster.NumThreads)==0, ...
        'ProcsPerNode (%d) must be greater than or equal to and divisible by NumThreads (%d).', ppn, cluster.NumThreads);
    commonSubmitArgs = sprintf('%s --ntasks-per-node=%d', commonSubmitArgs,ppn/cluster.NumThreads);
end
commonSubmitArgs = sprintf('%s --ntasks-per-core=1', commonSubmitArgs);

%% REQUIRED

%MW: User gets entire node.  No need to request memory.
% Physical memory used by a single core

% Queue name
qn = validatedPropValue(ap, 'QueueName', 'char', '');
if isempty(qn)
    emsg = sprintf(['\n\t>> %% Must set QueueName.  E.g.\n\n', ...
                    '\t>> c = parcluster;\n', ...
                    '\t>> c.AdditionalProperties.QueueName = ''queue-name'';\n', ...
                    '\t>> c.saveProfile\n\n']);
    error(emsg)
else
    commonSubmitArgs = sprintf('%s -p %s', commonSubmitArgs, qn);
end

% Wall time
wt = validatedPropValue(ap, 'WallTime', 'char', '');
if isempty(wt)
    emsg = sprintf(['\n\t>> %% Must set WallTime.  E.g.\n\n', ...
                    '\t>> c = parcluster;\n', ...
                    '\t>> %% 5 hours\n', ...
                    '\t>> c.AdditionalProperties.WallTime = ''05:00:00'';\n', ...
                    '\t>> c.saveProfile\n\n']);
    error(emsg)
else
    commonSubmitArgs = sprintf('%s -t %s', commonSubmitArgs, wt);
end


%% OPTIONAL

% Constraint
cn = validatedPropValue(ap, 'Constraint', 'char', '');
if ~isempty(cn)
    commonSubmitArgs = sprintf('%s -C %s', commonSubmitArgs, cn);
end

% Reservation
res = validatedPropValue(ap, 'Reservation', 'char', '');
if ~isempty(res)
    commonSubmitArgs = sprintf('%s --reservation=%s', commonSubmitArgs, res);
end

% Email notification
ea = validatedPropValue(ap, 'EmailAddress', 'char', '');
if ~isempty(ea)
    commonSubmitArgs = sprintf('%s --mail-type=ALL --mail-user=%s', commonSubmitArgs, ea);
end

% Catch-all
asa = validatedPropValue(ap, 'AdditionalSubmitArgs', 'char', '');
if ~isempty(asa)
    commonSubmitArgs = sprintf('%s %s', commonSubmitArgs, asa);
end

commonSubmitArgs = strtrim(commonSubmitArgs);

end
